/************************************************************************
 *
 * \file: mspin_demo_monitor.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: mySPIN - Demo application
 *
 * \author:
 *      Bui Le Thuan / RBVH/ENG2 / Thuan.BuiLe@vn.bosch.com
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include "mspin_demo_monitor.h"
#include <mspin_logging.h>

bool DeviceMonitor::m_EAPSessionStarted = false;
bool DeviceMonitor::m_iOSDeviceReady = false;

#ifndef MSPIN_IAP2_SUPPORT_DISABLED
timer_t DeviceMonitor::m_iAP2TimerId;
sem_t DeviceMonitor::m_iAP2ConnLock;
#endif //#ifndef MSPIN_IAP2_SUPPORT_DISABLED

//using namespace adit::uspi;

DeviceMonitor::DeviceMonitor()
{
    m_MonitorThread = 0;
    m_SharedMonCtx = nullptr;
    m_testbed_instance = nullptr;
    discoverer = nullptr;

    pSwitchedUsbDevice = nullptr;
}

DeviceMonitor::~DeviceMonitor()
{

}

void DeviceMonitor::setTestbedInstance(Testbed* instance)
{
    m_testbed_instance = instance;
}

Testbed* DeviceMonitor::getTestbedInstance()
{
    return m_testbed_instance;
}

void DeviceMonitor::setMonitorContext(mspin_demo_monitor_context_t* pInputMonCtx)
{
    m_SharedMonCtx = pInputMonCtx;
}

mspin_demo_monitor_context_t* DeviceMonitor::getMonitorContext()
{
    return m_SharedMonCtx;
}

usb_device DeviceMonitor::getFirstItemOfUsbDeviceList()
{
    return m_UsbDeviceList.front();
}

void DeviceMonitor::removeFirstItemOfUsbDeviceList()
{
    m_UsbDeviceList.pop_front();
}

bool DeviceMonitor::addItemIntoUsbDeviceList(usb_device inUsbDevice)
{
    try
    {
        m_UsbDeviceList.push_back(inUsbDevice);
        return true;
    }
    catch(std::bad_alloc& exc)
    {
        mspin_log_printLn(eMspinVerbosityError,
                            "%s() ERROR: Add suitable device in list failed", __FUNCTION__);
        return false;
    }

}

uint32_t DeviceMonitor::getSizeOfUsbDeviceList()
{
    return m_UsbDeviceList.size();
}

void DeviceMonitor::setSwitchedUsbDevice(std::shared_ptr<DiscoveredDeviceUsb> pInUsbDevice)
{
    pSwitchedUsbDevice = pInUsbDevice;
}

std::shared_ptr<DiscoveredDeviceUsb> DeviceMonitor::getSwitchedUsbDevice()
{
    return pSwitchedUsbDevice;
}

bool DeviceMonitor::isIOSDevice(uint32_t vendorId, uint32_t productId)
{
    if ((MYSPIN_DEMO_APPL_APPLE_VENDOR_ID == vendorId)
            && (productId >= MYSPIN_DEMO_APPL_APPLE_PRODUCT_ID_MIN)
            && (productId <= MYSPIN_DEMO_APPL_APPLE_PRODUCT_ID_MAX))
    {
        return TRUE;
    }
    else
    {
        return FALSE;
    }
}

bool DeviceMonitor::isIOSDeviceOnOTGPort(std::string devPath)
{
    bool rc = FALSE;
    std::string subDevPath;

    std::size_t found = devPath.find_last_of('/');
    if(found!=std::string::npos)
    {
        subDevPath = devPath.substr(found+1);
        if(subDevPath.compare(MSPIN_STR_OTG_PORT)==0)
        {
            rc = TRUE;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityVerbose,"Plugged port isn't OTG port.(path = %s)", devPath);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn,"Path is invalid (path = %s)", devPath);
    }

    return rc;
}

void DeviceMonitor::printUsbDeviceList()
{
    int ind = 0;

    if(m_UsbDeviceList.empty())
    {
        // list is empty.
        return;
    }

    for (std::list<usb_device>::iterator item = m_UsbDeviceList.begin(); item != m_UsbDeviceList.end(); ++item)
    {
        ind++;
        mspin_log_printLn(eMspinVerbosityVerbose,
                            "%s() index = %d - vendorId:%.4x - productId:%.4x - s/n='%s' - devPath='%s'",
                            __FUNCTION__, ind, (*item).vendorId, (*item).productId,
                            (*item).serial.c_str(), (*item).devPath.c_str());
    }
}

void DeviceMonitor::resetUsbDeviceList()
{
    m_UsbDeviceList.clear();
}

bool DeviceMonitor::start_monitor(U32 inDemoTimeout)
{
    uint32_t inEventMask = DD_USB_AOAP | DD_USB_APPLE;

    if(inDemoTimeout > 0)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s() Use only Android phone for automation test", __FUNCTION__);
        inEventMask = DD_USB_AOAP;
    }

    discoverer = new FeatureDiscovery(this, inEventMask);

    if (DiscoveryError::OK != discoverer->start())
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() Failed to start FeatureDiscovery", __FUNCTION__);
        discoverer = nullptr;
        return false;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s()  Use udevadm to trigger add event(s)", __FUNCTION__);
        int32_t err = system("udevadm trigger --type=devices --subsystem-match=usb --action=add");
        if (err != 0)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() Failed to trigger add event(s). system() returned with err = %d",
                    __FUNCTION__, err);
        }
    }

    return true;
}

void DeviceMonitor::stop_monitor()
{
    if (discoverer != nullptr)
    {
        if (DiscoveryError::OK != discoverer->stop())
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() Failed to properly stop FeatureDiscovery");
            // just continue though
        }
        delete discoverer;
        discoverer = nullptr;
    }
}

void DeviceMonitor::handle_found_device(DeviceMonitor* pArg, std::shared_ptr<DiscoveredDevice> inDevice)
{
    DeviceMonitor* pMonitorInstance = pArg;

    usb_device suitableDevice;
    uint32_t vendorId = 0;
    uint32_t productId = 0;
    std::string pSerial;
    std::string pDevPath;
    std::string pProduct;
    uint32_t devNum = 0;
    bool appNotRun = false;

    mspin_demo_monitor_context_t* pSharedMonCtx = pMonitorInstance->getMonitorContext();
    if(pMonitorInstance->getSizeOfUsbDeviceList() == 0)
    {
        // mySPIN demo application did not work when list of device is empty
        appNotRun = true;
    }

    std::shared_ptr<DiscoveredDeviceUsb> pInUsbDevice = std::dynamic_pointer_cast<DiscoveredDeviceUsb>(inDevice);
    DeviceInfo ddInfo = pInUsbDevice->getInfo();

    mspin_log_printLn(eMspinVerbosityVerbose, "%s() Handling found device", __FUNCTION__);

    vendorId = ddInfo.getiDeviceInfo(DSYS_IDVENDOR);
    productId = ddInfo.getiDeviceInfo(DSYS_IDPRODUCT);

    if (vendorId && productId)
    {
        pSerial  = ddInfo.getDeviceInfo(DSYS_SERIAL);
        pProduct = ddInfo.getDeviceInfo(DSYS_PRODUCT);
        pDevPath = ddInfo.getDevPath();
        devNum   = ddInfo.getiDeviceInfo(DSYS_DEVNUM);

        mspin_log_printLn(eMspinVerbosityInfo,
                "%s() device %.4x:%.4x w/serial='%s' devPath='%s' product='%s' attached",
                __FUNCTION__, vendorId, productId, pSerial.c_str(), pDevPath.c_str(), pProduct.c_str());

        if (!pSerial.empty())
        {
            if(pMonitorInstance->preselectDevice(vendorId, productId, pSerial, pDevPath))
            {
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s() Device candidate found with %.4x:%.4x w/s/n='%s'(len=%d) devNum=%d",
                        __FUNCTION__, vendorId, productId, pSerial.c_str(), pSerial.length(), devNum);

                suitableDevice.vendorId  = vendorId;
                suitableDevice.productId = productId;
                suitableDevice.devNum    = devNum;
                suitableDevice.serial    = pSerial;
                suitableDevice.devPath   = pDevPath;

                if(pMonitorInstance->addItemIntoUsbDeviceList(suitableDevice) && appNotRun)
                {
                    sem_post(&pSharedMonCtx->gConnectSemaphore);
                }
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityInfo,
                        "%s() Ignore this unsuitable device", __FUNCTION__);
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityInfo,
                    "%s() Ignore this device without serial", __FUNCTION__);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn,
                "%s() WARN: Ignore this device with invalid vendorId or productId", __FUNCTION__);
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s() Number of suitable device : %d",
            __FUNCTION__, pMonitorInstance->getSizeOfUsbDeviceList());
}

bool DeviceMonitor::preselectDevice(uint32_t vendorId, uint32_t productId, std::string serial, std::string devPath)
{
    bool connect = FALSE;

    if (serial.empty())
    {
        mspin_log_printLn(eMspinVerbosityWarn,
                "%s(%.4x:%.4x, s/n='%s', devPath='%s') WARNING: Serial is empty",
                __FUNCTION__, vendorId, productId, serial.c_str(), devPath.c_str());
        return connect;
    }

    switch (vendorId)
    {
        case 0x0bb4: //HTC
        case 0x04e8: //Samsung
        case 0x1004: //LG
        case 0x18d1: //Google
        case 0x0fce: //Sony Ericsson
        case 0x22b8: //Motorola
        //Add here additional vendor IDs if required
        {
            connect = TRUE;
            if(Testbed::instance().getTestbedMode())
            {
                Testbed::instance().setConnectedAndroidPhone(TRUE);
            }
            break;
        }
        case 0x05AC: //Apple
        {
            if (isIOSDevice(vendorId, productId))
            {
                if(m_testbed_instance->getTestbedMode())
                {
                    m_testbed_instance->setConnectedAndroidPhone(FALSE);
                }

                connect = isIOSDeviceOnOTGPort(devPath);
                if (!connect)
                {
                    mspin_log_printLn(eMspinVerbosityWarn,
                            "%s(%.4x:%.4x, s/n='%s', devPath='%s') WARNING: iOS device on non OTG port found -> ignore",
                            __FUNCTION__, vendorId, productId, serial.c_str(), devPath.c_str());
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityDebug,
                            "%s(%.4x:%.4x, s/n='%s', devPath='%s') connect to device at OTG port",
                            __FUNCTION__, vendorId, productId, serial.c_str(), devPath.c_str());
                }
            }
            else
            {
                connect = FALSE;
            }
            break;
        }
        default:
        {
            connect = FALSE;
            break;
        }
    }

    return connect;
}

void DeviceMonitor::OnUDPBroadcastEnd_CB(MSPIN_UDP_BROADCAST_END_REASON reason, void* context)
{
    mspin_log_printLn(eMspinVerbosityDebug,
            "%s **************More parameters************",
            __FUNCTION__);

    (void)reason;
    (void)context;
}

bool DeviceMonitor::OnAcceptIncomingConnection_CB(S32 connectionID, const U8* ipAddr, void* context)
{
    mspin_log_printLn(eMspinVerbosityDebug,
            "%s connectionID: %d **************More parameters************",
            __FUNCTION__);

    mspin_demo_connectionParameter_t *pDevice = (mspin_demo_connectionParameter_t*)context;
    if (!pDevice || !pDevice->pMonitorContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                    "%s(ctx=%p) ERROR: context or monitor context is NULL",
                    __FUNCTION__, context);
        return false;
    }

    pDevice->connectionID = connectionID;
    MSPIN_StopUDPBroadcasting();

    // signal to main thread
    std::unique_lock<std::mutex> lk(pDevice->pMonitorContext->g_wifi_monitor_m);
    pDevice->pMonitorContext->g_connected_device = true;
    lk.unlock();

    pDevice->pMonitorContext->g_wifi_monitor_cv.notify_one();

    mspin_log_printLn(eMspinVerbosityWarn,
                    "%s: Wifi connection is accepted", __FUNCTION__);

    (void)ipAddr;
    return true;
}

void DeviceMonitor::OnConnectionClosed_CB(S32 connectionID, MSPIN_TCP_CONNECTION_END_REASON reason, const U8* ipAddr, void* context)
{
    mspin_log_printLn(eMspinVerbosityDebug,
            "%s **************More parameters************",
            __FUNCTION__);

    mspin_demo_connectionParameter_t *pDevice = (mspin_demo_connectionParameter_t*)context;
    if (!pDevice || !pDevice->pMonitorContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                    "%s(ctx=%p) ERROR: context or monitor context is NULL",
                    __FUNCTION__, context);
        return;
    }

    pDevice->pMonitorContext->gQuit = TRUE;

    // signal to main thread
    std::unique_lock<std::mutex> lk(pDevice->pMonitorContext->g_wifi_monitor_m);
    pDevice->pMonitorContext->g_disconnected_device = true;
    lk.unlock();

    pDevice->pMonitorContext->g_wifi_monitor_cv.notify_one();

    mspin_log_printLn(eMspinVerbosityWarn,
                    "%s: Wifi connection is closed", __FUNCTION__);

    (void)connectionID;
    (void)reason;
    (void)ipAddr;
}

/*
 * iAP2 Monitor
 */

#ifndef MSPIN_IAP2_SUPPORT_DISABLED
void DeviceMonitor::iAP2DeviceReady_CB(void *pToken, bool ready)
{
    //Callback when iOS device is ready to be used (iAP2 connection established)

    m_iOSDeviceReady = ready;

    if (NULL != m_iAP2TimerId)
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(token=%p, ready=%s) device ready -> stop timer id=0x%1x",
                __FUNCTION__, pToken, ready ? "ready" : "not ready", (long)m_iAP2TimerId);
        DemoTimer::timer_stop(m_iAP2TimerId);
        m_iAP2TimerId = NULL;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn,
                "%s(token=%p, ready=%s) device ready -> timer isn't running, nothing to stop",
                __FUNCTION__, pToken, ready ? "ready" : "not ready");
    }

    sem_post(&(m_iAP2ConnLock));
}

void DeviceMonitor::iAP2DeviceDisconnected_CB(void *pToken)
{
    mspin_demo_connectionParameter_t *pDevice = (mspin_demo_connectionParameter_t*)pToken;
    if (!pDevice)
    {
        mspin_log_printLn(eMspinVerbosityError,
                    "%s(token=%p) ERROR: token could not be casted to RunDeviceArg_t",
                    __FUNCTION__, pToken);
        return;
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(token=%p) iOS device disconnected -> exit run loop",
            __FUNCTION__, pToken);

    m_iOSDeviceReady = FALSE;
    m_EAPSessionStarted = FALSE;

    pthread_mutex_lock(&(pDevice->switchStateMutex));
    pDevice->switchRunState = eMSPIN_DEMO_STATE_SWITCH_TO_EXIT;
    pthread_mutex_unlock(&(pDevice->switchStateMutex));
}

void DeviceMonitor::EAPSessionStart_CB(void *pToken)
{
    mspin_demo_connectionParameter_t *pDevice = (mspin_demo_connectionParameter_t*)pToken;
    if (!pDevice)
    {
        mspin_log_printLn(eMspinVerbosityError,
                    "%s(token=%p) ERROR: Failed to cast token to 'mspin_demo_deviceArguments_t*'",
                    __FUNCTION__, pToken);
        return;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(token=%p) called", __FUNCTION__, pToken);
    }

    m_EAPSessionStarted = TRUE;
    pthread_mutex_lock(&(pDevice->switchStateMutex));
    mspin_log_printLn(eMspinVerbosityInfo,
            "%s(token=%p) mySPIN Launcher app started (in state='%s' and switching state='%s')-> start mySPIN Core",
            __FUNCTION__, pToken, getRunState(pDevice->runState).c_str(),
            getSwitchRunState(pDevice->switchRunState).c_str());
    pDevice->switchRunState = eMSPIN_DEMO_STATE_SWITCH_TO_RUN;
    pthread_mutex_unlock(&(pDevice->switchStateMutex));
}

void DeviceMonitor::EAPSessionStop_CB(void *pToken)
{
    mspin_demo_connectionParameter_t *pDevice = (mspin_demo_connectionParameter_t*)pToken;
    if (!pDevice)
    {
        mspin_log_printLn(eMspinVerbosityError,
                    "%s(token=%p) ERROR: token could not be casted to RunDeviceArg_t",
                    __FUNCTION__, pToken);
        return;
    }

    m_EAPSessionStarted = FALSE;
    if (eMSPIN_DEMO_STATE_RUNNING == pDevice->runState)
    {
        pthread_mutex_lock(&(pDevice->switchStateMutex));
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(token=%p) mySPIN Launcher app stopped (in state='%s')-> stop mySPIN Core",
                __FUNCTION__, pToken, getRunState(pDevice->runState).c_str());
        pDevice->switchRunState = eMSPIN_DEMO_STATE_SWITCH_TO_STOP;
        pthread_mutex_unlock(&(pDevice->switchStateMutex));
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(token=%p) mySPIN Launcher app stopped but we are already in state='%s'-> ignore",
                __FUNCTION__, pToken, getRunState(pDevice->runState).c_str());
    }
}

void DeviceMonitor::connectTimeoutHandler_CB(int sig, siginfo_t *si, void *uc)
{
    m_iOSDeviceReady = FALSE;
    m_iAP2TimerId = NULL;

    if (!m_iOSDeviceReady)
    {
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(sig=%d,si=0x%x, uc=0x%x) device not yet ready -> cancel iAP2 connect",
                __FUNCTION__, sig, si, uc);

        sem_post(&(m_iAP2ConnLock));
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(sig=%d,si=0x%x, uc=0x%x) device is already ready -> ignore",
                __FUNCTION__, sig, si, uc);
    }
}

MSPIN_ERROR DeviceMonitor::connectIAP2(mspin_demo_connectionParameter_t *pDevice)
{
    if (!pDevice)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(device=%p) FATAL ERROR: device is NULL",
                __FUNCTION__, pDevice);
        return MSPIN_ERROR_GENERAL;
    }

    m_iOSDeviceReady = FALSE; //reset static variable

    //Init semaphore
    (void)sem_init(&(m_iAP2ConnLock), 0, 0);  //binary semaphore locked

    //Start timer here to cancel iAP2 connection attempt
    m_iAP2TimerId = DemoTimer::timer_start(connectTimeoutHandler_CB, 15, 0); //15 second timer

    mspin_log_printLn(eMspinVerbosityInfo,
            "%s(device=%p) timerId=0x%1x started -> connect now",
            __FUNCTION__, pDevice, (long)m_iAP2TimerId);

    if (0 != m_iAP2_Connection.connect(pDevice->serial, iAP2DeviceReady_CB,
            iAP2DeviceDisconnected_CB, EAPSessionStart_CB,
            EAPSessionStop_CB, pDevice))
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(device=%p) ERROR: Failed to connect to device",
                __FUNCTION__, pDevice);
        return MSPIN_ERROR_CONNECTION_START;
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(device=%p) -> waiting for device to become ready",
            __FUNCTION__, pDevice);

    //Wait until iAP2 has connected the phone and the device becomes ready
    (void)sem_wait(&(m_iAP2ConnLock));

    if (m_iOSDeviceReady)
    {
        if (isIOSDevice(pDevice->vendorId, pDevice->productId))
        {
            m_iAP2_Connection.powerSourceUpdate();

            pDevice->runState = eMSPIN_DEMO_STATE_STOPPED;
            if (m_EAPSessionStarted)
            {
                mspin_log_printLn(eMspinVerbosityInfo,
                        "%s(device=%p) device ready and mySPIN Launcher already running -> switch immediately to run state",
                        __FUNCTION__, pDevice);
                pthread_mutex_lock(&(pDevice->switchStateMutex));
                pDevice->switchRunState = eMSPIN_DEMO_STATE_SWITCH_TO_RUN;
                pthread_mutex_unlock(&(pDevice->switchStateMutex));
            }
            else
            {
                //Request to start launcher app and wait for Launcher app
                mspin_log_printLn(eMspinVerbosityInfo,
                        "%s(device=%p) device ready -> try to request app launch. Otherwise launch application manually",
                        __FUNCTION__, pDevice);

#ifdef MSPIN_DEMO_SHOW_APP_LAUNCH_POPUP
                m_iAP2_Connection.requestAppLaunch(TRUE);
#else
                m_iAP2_Connection.requestAppLaunch(FALSE);
#endif //MSPIN_DEMO_SHOW_APP_LAUNCH_POPUP
            }
        }

        if(Testbed::instance().getTestbedMode())
        {
            Testbed::instance().setApplePhoneDisconnectionStatus(FALSE);
        }

        return MSPIN_SUCCESS;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(device=%p) ERROR: device not ready",
                __FUNCTION__, pDevice);
        return MSPIN_ERROR_CONNECTION_START;
    }
}

void DeviceMonitor::disconnectIAP2()
{
    m_iAP2_Connection.disconnect();
}

#endif //#ifndef MSPIN_IAP2_SUPPORT_DISABLED


std::string DeviceMonitor::getRunState(mspin_demo_run_state_t state)
{
    switch (state)
    {
        case eMSPIN_DEMO_STATE_RUNNING:
        {
            return "RUNNING";
            //break;
        }
        case eMSPIN_DEMO_STATE_STOPPED:
        {
            return "STOPPED";
            //break;
        }
        default:
        {
            return "UNKNOWN";
            //break;
        }
    }
}

std::string DeviceMonitor::getSwitchRunState(mspin_demo_switch_run_state_t state)
{
    switch (state)
    {
        case eMSPIN_DEMO_STATE_STABLE:
        {
            return "STABLE";
            //break;
        }
        case eMSPIN_DEMO_STATE_SWITCH_TO_RUN:
        {
            return "SWITCH_TO_RUN";
            //break;
        }
        case eMSPIN_DEMO_STATE_SWITCH_TO_STOP:
        {
            return "SWITCH_TO_STOP";
            //break;
        }
        case eMSPIN_DEMO_STATE_SWITCH_TO_EXIT:
        {
            return "SWITCH_TO_EXIT";
            //break;
        }
        default:
        {
            return "UNKNOWN";
            //break;
        }
    }
}


DiscoveryError DeviceMonitor::foundCb(std::shared_ptr<DiscoveredDevice> inDevice)
{
    DiscoveryError res = DiscoveryError::OK;

    if (nullptr != inDevice)
    {
        handle_found_device(this, inDevice);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() Error: device object is invalid", __FUNCTION__);
    }

    return res;
}

DiscoveryError DeviceMonitor::lostCb(std::shared_ptr<DiscoveredDevice> inDevice)
{
    DiscoveryError res = DiscoveryError::OK;

    if (nullptr != inDevice)
    {
        if(m_testbed_instance->getTestbedMode())
        {
            if(m_testbed_instance->getState() != TBA_STATE_FOREGROUND)
            {
                m_testbed_instance->adit_testbed_lost();

                if (m_testbed_instance->getConnectedAndroidPhone())
                {
                    m_testbed_instance->setAndroidPhoneDisconnectionStatus(TRUE);
                    mspin_log_printLn(eMspinVerbosityInfo, "Android phone is disconnected.\n");
                }
                else
                {
                    m_testbed_instance->setApplePhoneDisconnectionStatus(TRUE);
                    mspin_log_printLn(eMspinVerbosityInfo, "Apple phone is disconnected.\n");
                }
            }
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() Error: device info is null", __FUNCTION__);
    }

    return res;
}

DiscoveryError DeviceMonitor::switchedCb(std::shared_ptr<DiscoveredDevice> inDevice)
{
    DiscoveryError res = DiscoveryError::OK;

    if (nullptr != inDevice)
    {
        std::shared_ptr<DiscoveredDeviceUsb> pInUsbDevice = std::dynamic_pointer_cast<DiscoveredDeviceUsb>(inDevice);
        DeviceInfo ddInfo = pInUsbDevice->getInfo();

        const char *productIdString = NULL;
        unsigned int productId = 0;

        productIdString = ddInfo.getDeviceInfo(DSYS_IDPRODUCT).c_str();
        if (productIdString)
        {
            sscanf(productIdString, "%x", &productId);

            if ((productId >= 0x2d00) && (productId <= 0x2d05))
            {
                setSwitchedUsbDevice(pInUsbDevice);
                mspin_log_printLn(eMspinVerbosityInfo, "%s()  Store switched device", __FUNCTION__);
            }
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s()  Error: device info is null", __FUNCTION__);
    }

    return res;
}

DiscoveryError DeviceMonitor::changedCb(std::shared_ptr<DiscoveredDevice> inDevice)
{
    DiscoveryError res = DiscoveryError::OK;

    if (nullptr != inDevice)
    {

    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "DeviceMonitor::%s()  Error: device info is null", __FUNCTION__);
    }

    return res;
}

DiscoveryError DeviceMonitor::errorCb(DiscoveryError inErrorCode)
{
    mspin_log_printLn(eMspinVerbosityError, "%s() Error %s occurred", __FUNCTION__, to_str(inErrorCode));
    return DiscoveryError::OK;
}
